/* ================================================================
 * ================================================================
 *
 * A basic robot controller for the footbot.
 * Includes mostly movement and vector calculation functions
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */

#include "baseRobot.h"

#include <algorithm>
#include <math.h>
#include <string>
#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/core/utility/math/vector2.h>

#include "gwrNN.h"
#include "../helpers/constants.h"
#include "../helpers/logger.h"

BaseRobot::BaseRobot() {
   numId = -1;
	wheels = NULL;
	lightSensor = NULL;
	proximitySensor = NULL;
	blobSensor = NULL;
	leds = NULL;
	groundSensor = NULL;
   CRadians collisionAvoidingAngle = CRadians(0);
   currentWaypoint = NULL;
   gwrNN = new GWRNN();
}

/**
 * Initialise
 */
void BaseRobot::Init(TConfigurationNode& t_node) {

   try {
      //-- create hardware representations
      wheels = GetActuator<CCI_DifferentialSteeringActuator>("differential_steering");
      proximitySensor = GetSensor<CCI_FootBotProximitySensor>("footbot_proximity");
      lightSensor = GetSensor<CCI_FootBotLightSensor>("footbot_light");
      leds = GetActuator<CCI_LEDsActuator>("leds");
      blobSensor = GetSensor<CCI_ColoredBlobOmnidirectionalCameraSensor>("colored_blob_omnidirectional_camera");
      groundSensor = GetSensor <CCI_FootBotMotorGroundSensor>("footbot_motor_ground");

      //-- parse XML parameters
      params_navigation.Init(GetNode(t_node, "navigation"));
      params_sensors.Init(GetNode(t_node, "sensors"));
      gwrNN->Init(GetNode(t_node,"gwrnn"));

   } catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing the foot-bot foraging controller for robot \"" << GetId() << "\"", ex);
   }

   //-- setup hardware
   blobSensor->Enable();

   //-- other

}

BaseRobot::~BaseRobot() {
   delete gwrNN;
}

/* ==================================================================================== */
/* ========================= BEHAVIOUR */
/* ==================================================================================== */

/**
 * One update loop step
 */
void BaseRobot::ControlStep() {

   //-- navigate between waypoints
   if (currentWaypoint != NULL) {
      if (NavigateTowardsWaypoint(currentWaypoint)) {
         SetWheelSpeeds(0,0);
         OnWaypointReached();
      }
   } else {
      SetWheelSpeeds(0,0);
   }


   //-- do novelty detection
   std::vector<float> blobSensorDataArray;
   GetBlobSensorData(blobSensorDataArray);
   if (numId == 0) {
      //Logger::Out("DA", blobSensorDataArray);
   }

   gwrNN->ProcessInput(blobSensorDataArray);

   //-- indicate the amount of currently perceived novelty with LEDs
   if (gwrNN->GetNoveltyValue() >= 0.9) {
      leds->SetAllColors(CColor::RED);
   /*} else if (gwrNN->GetNoveltyValue() > 0.4) {
      leds->SetAllColors(CColor::YELLOW); */
   } else {
      leds->SetAllColors(CColor::BLACK);
   }


}

/**
 * Trigerred when a waypoint was reached.
 * Switch to the next waypoint or loop around the waypoint array.
 */
void BaseRobot::OnWaypointReached() {
   //-- find the next waypoint, wrapping around the array
   currentWaypoint = currentWaypoint->nextWaypoint;
   if (currentWaypoint == NULL) {
      currentWaypoint = waypoints[0];
   }
}

/* ==================================================================================== */
/* ========================= MEMORY */
/* ==================================================================================== */

/**
 * Add a waypoint to the list of remembered waypoits
 */
void BaseRobot::AddWaypoint(const Waypoint& waypoint_) {
   //LOG << "Robot " << numId << " got WP " << waypoint_.id << " at [" << waypoint_.position.GetX() << "," << waypoint_.position.GetY() << "]" << std::endl;
   Waypoint* newWp = new Waypoint(waypoint_);
   waypoints.push_back(newWp);

   //-- link the previous waypoint to point to this one
   if (waypoints.size() >= 2) {
      waypoints[waypoints.size()-2]->nextWaypoint = newWp;
   }
   //-- if this is the 1st waypoint, make it current
   if (waypoints.size() == 1) {
      currentWaypoint = newWp;
   }
}

/**
 * Remember unique id this robot is identified by
 */
void BaseRobot::SetId(int id_) {
   numId = id_;
   gwrNN->SetRobotId(id_);
}

/* ==================================================================================== */
/* ========================= SENSORS */
/* ==================================================================================== */

/**
 * Get all perceived color blobs into a 1D array, with normalised values for distances and angles.
 * Sorted by blob distance.
 * Only returns first N blobs if the GWRNN has non-adaptive number of in weights
 */
void BaseRobot::GetBlobSensorData(std::vector<float>& dataArray_) {
   size_t i;
   const CCI_ColoredBlobOmnidirectionalCameraSensor::SReadings& data = blobSensor->GetReadings();
   std::vector<CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob> orderedData;
   if(!data.BlobList.empty()) {
      for(i = 0; i < data.BlobList.size(); ++i) {
         if (data.BlobList[i]->Distance <= params_sensors.colorBlobMaxRange && data.BlobList[i]->Angle.GetAbsoluteValue()/ M_PI <= (params_sensors.colorBlobMaxAngle/180)) {
            //LOGERR << GetColorValue(data.BlobList[i]->Color) << " " << data.BlobList[i]->Distance << "  " << data.BlobList[i]->Angle.GetValue() << std::endl;
            //LOGERR << GetColorValue(data.BlobList[i]->Color) << " " << data.BlobList[i]->Distance / params_sensors.colorBlobMaxRange << "  " << data.BlobList[i]->Angle.GetValue() / M_PI << std::endl;
            //-- create a new copy of the blob data, because the data can change outside this loop (e.g. if blob becomes unavailable)
            CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob blob(data.BlobList[i]->Color,data.BlobList[i]->Angle,data.BlobList[i]->Distance);
            orderedData.push_back(blob);
         }
      }

      //-- sort the blobs by distance. Blobs of same distance will be sorted randomly.
      std::sort(orderedData.begin(), orderedData.end(), [this] (CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob a, CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob b) {
         if (a.Distance == b.Distance) {
            return (Helpers::GetRandomFloat(0, 1) > 0.5);
         } else {
            return a.Distance < b.Distance;
         }
      });

      //-- create the data array by copying and normalising the blob values
      int maxInputs = orderedData.size();
      if (!gwrNN->GetIsNumOfInWeightsAdaptive()) {
         maxInputs = Min((int)(orderedData.size()), gwrNN->GetMaxInputSize() / 3); // because there are 3 in weights per blob
      }
      for (i = 0; i < maxInputs; i++) {
         dataArray_.push_back(GetColorValue(orderedData[i].Color));
         dataArray_.push_back(orderedData[i].Distance / params_sensors.colorBlobMaxRange);
         dataArray_.push_back(orderedData[i].Angle.GetValue() / M_PI);
      }
      if (!gwrNN->GetIsNumOfInWeightsAdaptive()) {
         if (dataArray_.size() < gwrNN->GetMaxInputSize()) {
            for (i=dataArray_.size(); i< gwrNN->GetMaxInputSize(); i++) {
               dataArray_.push_back(0);
            }
         }
      }
   }
}


/**
 * Return color representation as a number, calculated as:
 *    c = 1/6*r + 2/6*g + 3/6*b
 * Where R,G,B are channel values between 0 and 1 and c is therefore between 0 and 1. Each combination of R, G, B values is unique.
 * Additionally, Gaussian noize is added to each of the R,G,B values using blobCameraNoizeStdDev std dev.
 */
float BaseRobot::GetColorValue(CColor color_) {
   //-- add noise to channels
   float red = color_.GetRed()/255.0 + Helpers::GetRandomGaussian(params_sensors.blobCameraNoizeStdDev);
   float green = color_.GetGreen()/255.0 + Helpers::GetRandomGaussian(params_sensors.blobCameraNoizeStdDev);
   float blue = color_.GetBlue()/255.0 + Helpers::GetRandomGaussian(params_sensors.blobCameraNoizeStdDev);

   //-- calculate the color value
   return 1/6.0*red + 2/6.0*green + 3/6.0*blue;
}

/**
 * Comparison function used for sorting color blobs
 */
bool BaseRobot::CompareSBlobsByDistance(CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob* a, CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob* b) {
    return a->Distance < b->Distance;
}

/* ==================================================================================== */
/* ========================= OUTPUT */
/* ==================================================================================== */

/**
 * Get a debug string that gets rendered in the on-screen overlay (see also loop_functions/qtLoopFunctions.h)
 */
std::string BaseRobot::GetOnScreenDebugString() {
   std::string debugText = GetId();
   debugText += " WN " + std::to_string(gwrNN->GetWinningNeuronNumber()) + " nov " + std::to_string(gwrNN->GetNoveltyValue());

   //-- actual position
   CDegrees deg;
   deg.FromValueInRadians(rotationTruth.GetValue());
   debugText += "\n[" + Helpers::FloatToString(positionTruth.GetX()) + ";" + Helpers::FloatToString(positionTruth.GetY());
   debugText += "] R=" + Helpers::FloatToString(deg.GetValue());

   //-- believed position
   debugText += "\n{" + Helpers::FloatToString(positionEstimate.GetX()) + ";" + Helpers::FloatToString(positionEstimate.GetY()) + "}";

   return debugText;
}

/* ==================================================================================== */
/* ========================= MOVEMENT */
/* ==================================================================================== */

/**
 * Navigate towards waypoint.
 * Returns true if estimating that at the waypoint, otherwise false.
 */
bool BaseRobot::NavigateTowardsWaypoint(Waypoint* wp_) {
   //-- find out if at the deposit already
   if (GetIsAtWaypoint(wp_)) {
      return true;
   }
   NavigateTowardsVector(wp_->GetPosition2D());
   return false;
}

/**
 * Returns true if estimating that is at the waypoint, otherwise false.
 */
bool BaseRobot::GetIsAtWaypoint(Waypoint* wp_) {
   const CCI_FootBotMotorGroundSensor::TReadings& groundSensorData = groundSensor->GetReadings();
   if (groundSensorData[0].Value <= WAYPOINT_GROUND_COLOR && groundSensorData[2].Value <= WAYPOINT_GROUND_COLOR) {
      if ((positionEstimate - wp_->position).Length() <= WAYPOINT_RADIUS) {
         return true;
      }
   }
   return false;
}

/**
 * Navigates towards a vector (in absolute coordinates), avoiding objects at the same time.
 * Returns true if collision detected.
 */
bool BaseRobot::NavigateTowardsVector(const CVector2& vector_) {
   bool collision = UpdateDiffusionVector();
   //LOG << this->GetId() << "Going to vector" << vector_ << std::endl;
   if (!collision) {
      SetWheelSpeedsFromAbsoluteVector(vector_);
   } else {
      //-- turn away rapidly
      SetWheelSpeedsFromDiffusionVector(params_navigation.maxSpeed*diffusionVector);
   }
   return collision;
}

/**
 * Set wheel speed based towards an absolute position in the world
 */
void BaseRobot::SetWheelSpeedsFromAbsoluteVector(const CVector2& absVector_) {
   CVector2 relVector = Helpers::GetRelativeVectorFromAnchor(GetPositionEstimate2D(), rotationEstimate.GetValue(), absVector_);
   CVector2 diff = absVector_ - GetPositionTruth2D();
   CVector2 relVector2 = CVector2(diff.Length(), relVector.Angle() - CRadians(M_PI/2));
   SetWheelSpeedsFromDiffusionVector(params_navigation.maxSpeed*relVector2*0.75 + params_navigation.maxSpeed*diffusionVector*1.25);
}

/*
 * Set wheel speed based on a relative vector, that is a result of Helpers::GetRelativeVectorFromAnchor.
 */
void BaseRobot::SetWheelSpeedsFromRelativeVector(const CVector2& relVector_) {

   //-- get angle towards the rel vector
   CRadians angle = CRadians(relVector_.GetX());
   CDegrees angleInDegrees;
   angleInDegrees.FromValueInRadians(angle.GetValue());

   //LOGERR << "Set wheel speeds from " << relVector_  << " A" << angle << std::endl;

   Real leftWheelSpeed, rightWheelSpeed;
   //-- if target in front, pick a random choice of how it will turn towards a target at the back next time
   if (angleInDegrees.GetValue() < 20 && relVector_.GetY() > 0 ) {
      params_navigation.rotateTowardsTargetBehindByLeft = (Helpers::GetRandomFloat(0, 1) > 0.5);
      ///LOGERR << " next time turning back to left:" << wheelParams.rotateTowardsTargetBehindByLeft << std::endl;
   } else if (params_navigation.angleToTargetChangeCounter > 100) {
      params_navigation.angleToTargetChangeCounter = 0;
      params_navigation.rotateTowardsTargetBehindByLeft = (Helpers::GetRandomFloat(0, 1) > 0.5);
      //LOGERR << " next time turning back to left:" << wheelParams.rotateTowardsTargetBehindByLeft << std::endl;
   } else {
      params_navigation.angleToTargetChangeCounter++;
   }

   if (relVector_.GetY() < 0 || angleInDegrees.GetValue() > 150 ) {
      //LOGERR << "TARGET BEHIND" << std::endl;
      //-- behind robot, rotate a lot
      if (params_navigation.rotateTowardsTargetBehindByLeft) {
         leftWheelSpeed = 0 *params_navigation.maxSpeed;
         rightWheelSpeed = 0.5* params_navigation.maxSpeed;
      } else {
         leftWheelSpeed = 0.5*params_navigation.maxSpeed;
         rightWheelSpeed = 0* params_navigation.maxSpeed;
      }
   } else if (angleInDegrees.GetAbsoluteValue() < 20) {
      //LOGERR << " target in front " << std::endl;
      leftWheelSpeed = params_navigation.maxSpeed;
      rightWheelSpeed = params_navigation.maxSpeed;

   } else if (angle.GetValue() > 0){
      //LOGERR << "target on right" << std::endl;
      if (angleInDegrees.GetAbsoluteValue() < 50) {
         //LOGERR << " soft " << std::endl;
         //-- soft turn
         Real fSpeedFactor = (params_navigation.hardTurnOnAngleThreshold - Abs(angle)) / params_navigation.hardTurnOnAngleThreshold;
         leftWheelSpeed = params_navigation.maxSpeed + params_navigation.maxSpeed * (1.0 - fSpeedFactor);
         rightWheelSpeed = params_navigation.maxSpeed - params_navigation.maxSpeed * (1.0 - fSpeedFactor);
      } else {
         //-- sharp turn
         leftWheelSpeed = 0.5*params_navigation.maxSpeed;
         rightWheelSpeed = -0.5*params_navigation.maxSpeed;
      }

   } else {
      //LOGERR << " target on left " << std::endl;
      if (angleInDegrees.GetAbsoluteValue() < 50) {
         //LOGERR << " soft " << std::endl;
         //-- soft turn
         Real fSpeedFactor = (params_navigation.hardTurnOnAngleThreshold - Abs(angle)) / params_navigation.hardTurnOnAngleThreshold;
         leftWheelSpeed = params_navigation.maxSpeed - params_navigation.maxSpeed * (1.0 - fSpeedFactor);
         rightWheelSpeed = params_navigation.maxSpeed + params_navigation.maxSpeed * (1.0 - fSpeedFactor);
      } else {
         //-- sharp turn
         leftWheelSpeed = -0.5*params_navigation.maxSpeed;
         rightWheelSpeed = 0.5*params_navigation.maxSpeed;
      }

   }
   SetWheelSpeeds(leftWheelSpeed, rightWheelSpeed);

}

/**
 * Set wheel speed based towards a vector that is specified by length and angle, as in
 * UpdateDiffusionVector()
 */
void BaseRobot::SetWheelSpeedsFromDiffusionVector(const CVector2& relVector_) {

   CRadians angle = relVector_.Angle().SignedNormalize();
   //LOGERR << "Set wheel speeds from diff " << relVector_  << " A" << angle << std::endl;

   //-- get the length of the relative vector
   Real fHeadingLength = relVector_.Length();

   //-- clamp the speed so that it's not greater than MaxSpeed
   //Real fBaseAngularWheelSpeed = Min<Real>(fHeadingLength, wheelParams.maxSpeed);
   Real fBaseAngularWheelSpeed = params_navigation.maxSpeed;

   //-- turning state switching conditions
   if(Abs(angle) > params_navigation.hardTurnOnAngleThreshold) {
      //-- hard Turn, heading angle very large
      params_navigation.turningMechanism = NavigationParams::HARD_TURN;
      //LOGERR << " > hard turn " << Abs(angle) << std::endl;
   } else if(Abs(angle) <= params_navigation.noTurnAngleThreshold) {
      //-- no Turn, heading angle very small
      params_navigation.turningMechanism = NavigationParams::NO_TURN;
      //LOGERR << " > too low " << Abs(angle) << std::endl;
   } else  { //if(wheelTurningParams.TurningMechanism == WheelTurningParams::NO_TURN && Abs(cHeadingAngle) > wheelTurningParams.SoftTurnOnAngleThreshold
      //-- soft Turn, heading angle in between the two cases
      params_navigation.turningMechanism = NavigationParams::SOFT_TURN;
      //LOGERR << " > soft turn " << Abs(angle) << std::endl;
   }

   //-- wheel speeds based on current turning state
   Real fSpeed1, fSpeed2;
   switch(params_navigation.turningMechanism) {
      case NavigationParams::NO_TURN: {
         fSpeed1 = fBaseAngularWheelSpeed;
         fSpeed2 = fBaseAngularWheelSpeed;
         break;
      }

      case NavigationParams::SOFT_TURN: {
         Real fSpeedFactor = (params_navigation.hardTurnOnAngleThreshold - Abs(angle)) / params_navigation.hardTurnOnAngleThreshold;
         fSpeed1 = fBaseAngularWheelSpeed - fBaseAngularWheelSpeed * (1.0 - fSpeedFactor);
         fSpeed2 = fBaseAngularWheelSpeed + fBaseAngularWheelSpeed * (1.0 - fSpeedFactor);
         break;
      }

      case NavigationParams::HARD_TURN: {
         fSpeed1 = 0.1*params_navigation.maxSpeed;
         fSpeed2 =  params_navigation.maxSpeed;
         break;
      }
   }


   //-- apply smoothing to alternating rotations
   if (params_navigation.angleToTargetChangeCounter <= 0) {
      if(angle > CRadians::ZERO) {
         params_navigation.rotateTowardsTargetBehindByLeft = true;
      } else {
         params_navigation.rotateTowardsTargetBehindByLeft = false;
      }
   } else {
      params_navigation.angleToTargetChangeCounter--;
   }

   if (angle.GetAbsoluteValue() < params_navigation.hardTurnOnAngleThreshold.GetAbsoluteValue()) {
      params_navigation.angleToTargetChangeCounter = 1;
   }

   //-- apply the calculated speeds to the appropriate wheels
   Real fLeftWheelSpeed, fRightWheelSpeed;
   if(params_navigation.rotateTowardsTargetBehindByLeft) {
      //-- turn left
      //LOGERR << "Turn left" << "|" << fSpeed1 << "|" << fSpeed2 << std::endl;
      fLeftWheelSpeed  = fSpeed1;
      fRightWheelSpeed = fSpeed2;

   } else {
      //-- turn right
      //LOGERR << "Turn right" << "|" << fSpeed2 << "|" << fSpeed1 << std::endl;
      fLeftWheelSpeed  = fSpeed2;
      fRightWheelSpeed = fSpeed1;
   }
   SetWheelSpeeds(fLeftWheelSpeed, fRightWheelSpeed);
}

/**
 * Sets wheel speed and remembers the decision
 */
void BaseRobot::SetWheelSpeeds(Real left_, Real right_) {
   wheels->SetLinearVelocity(left_ + Helpers::GetRandomGaussian(params_navigation.wheelNoizeStdDev), right_ + Helpers::GetRandomGaussian(params_navigation.wheelNoizeStdDev));
}


/*
 * Calculates the diffusion vector. If there is a close obstacle, it points away from it; it there is none, it points forwards.
 * @return bool whether collision avoidance occurred or not
 */
bool BaseRobot::UpdateDiffusionVector() {
   /* Get readings from proximity sensor */
   const CCI_FootBotProximitySensor::TReadings& tProxReads = proximitySensor->GetReadings();
   /* Sum them together */
   CVector2 cDiffusionVector;
   for(size_t i = 0; i < tProxReads.size(); ++i) {
      cDiffusionVector += CVector2(tProxReads[i].Value, tProxReads[i].Angle);
   }

   ////////////LOGERR << cDiffusionVector.Length() << cDiffusionVector.Angle() << std::endl;
   //-- if the angle of the vector is small enough and the closest obstacle is far enough, ignore the vector
   if(params_navigation.goStraightAngleRange.WithinMinBoundIncludedMaxBoundIncluded(cDiffusionVector.Angle()) && cDiffusionVector.Length() < params_navigation.delta ) {
      diffusionVector = CVector2::X;
      diffusionVector = -cDiffusionVector;
   } else {
      cDiffusionVector.Normalize();
      diffusionVector = -cDiffusionVector;
      return true;
   }
   return false;


}

/**
 * Returns true if an object was detected by proximity sensors
 */
bool BaseRobot::IsSomethingNearby() {
   const CCI_FootBotProximitySensor::TReadings& tProxReads = proximitySensor->GetReadings();
   for(size_t i = 0; i < tProxReads.size(); ++i) {
      ////////////LOGERR << tProxReads[i].Value << " :" << tProxReads[i].Angle << std::endl;
      if (tProxReads[i].Value > 0) {
         return true;
      }
   }
   return false;
}


/* ==================================================================================== */
/* ========================= PARAMETERS */
/* ==================================================================================== */


void BaseRobot::NavigationParams::Init(TConfigurationNode& t_node) {
   try {
      GetNodeAttribute(t_node, "wheelNoizeStdDev", wheelNoizeStdDev);
      GetNodeAttribute(t_node, "maxSpeed", maxSpeed);

      CDegrees cAngle;
      GetNodeAttribute(t_node, "hardTurnAngleThreshold", cAngle);
      hardTurnOnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "noTurnAngleThreshold", cAngle);
      noTurnAngleThreshold = ToRadians(cAngle);

      CRange<CDegrees> cGoStraightAngleRangeDegrees(CDegrees(-10.0f), CDegrees(10.0f));
      GetNodeAttribute(t_node, "goStraightAngleRange", cGoStraightAngleRangeDegrees);
      goStraightAngleRange.Set(ToRadians(cGoStraightAngleRangeDegrees.GetMin()),
                               ToRadians(cGoStraightAngleRangeDegrees.GetMax()));
      GetNodeAttribute(t_node, "delta", delta);
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initialising controller navigation parameters.", ex);
   }
}

void BaseRobot::SensorsParams::Init(TConfigurationNode& t_node) {
   try {
      GetNodeAttribute(t_node, "colorBlobMaxRange", colorBlobMaxRange);
      GetNodeAttribute(t_node, "colorBlobMaxAngle", colorBlobMaxAngle);
      GetNodeAttribute(t_node, "blobCameraNoizeStdDev", blobCameraNoizeStdDev);
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initialising controller sensors parameters.", ex);
   }
}


REGISTER_CONTROLLER(BaseRobot, "baseRobot")
