/* ================================================================
 * ================================================================
 *
 * A basic robot controller for the footbot.
 * Includes mostly movement and vector calculation functions
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */

#ifndef FOOTBOT_DIFFUSION_H
#define FOOTBOT_DIFFUSION_H


#include <argos3/core/control_interface/ci_controller.h>
#include <argos3/plugins/robots/generic/control_interface/ci_differential_steering_actuator.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_proximity_sensor.h>
#include <argos3/plugins/robots/generic/control_interface/ci_leds_actuator.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_light_sensor.h>
#include <argos3/plugins/robots/generic/control_interface/ci_colored_blob_omnidirectional_camera_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_motor_ground_sensor.h>
#include <argos3/core/utility/math/vector3.h>

#include "../entities/waypoint.h"

using namespace argos;

class GWRNN;

class BaseRobot : public CCI_Controller {

public:


   BaseRobot();
   virtual ~BaseRobot();


   virtual void Init(TConfigurationNode& t_node);
   virtual void ControlStep();
   virtual void Reset() {}
   virtual void Destroy() {}

   void SetId(int id_);
   int GetNumId() { return numId; }
   void SetPositionEstimate (const CVector3& newEstimate_) { positionEstimate.Set(newEstimate_.GetX(), newEstimate_.GetY(), newEstimate_.GetZ()); }
   void SetPositionTruth (const CVector3& new_) { positionTruth.Set(new_.GetX(), new_.GetY(), new_.GetZ()); }
   void SetRotationEstimate (CRadians newEstimate_) { rotationEstimate = newEstimate_; }
   void SetRotationTruth (CRadians new_) { rotationTruth = new_; }
   CVector3 GetPositionEstimate() { return positionEstimate; }
   CVector2 GetPositionEstimate2D() { return CVector2(positionEstimate.GetX(), positionEstimate.GetY()); }
   CVector3 GetPositionTruth() { return positionTruth; }
   CVector2 GetPositionTruth2D() { return CVector2(positionTruth.GetX(), positionTruth.GetY()); }
   CRadians GetRotationEstimate() { return rotationEstimate; }
   CRadians GetRotationTruth() { return rotationTruth; }

   CVector2 GetDiffusionVec() { return diffusionVector; }
   GWRNN* GetNetwork() { return gwrNN; }

   void AddWaypoint(const Waypoint& waypoint_);
   std::vector<Waypoint*>* GetWaypoints() { return &waypoints; }
   Waypoint* GetCurrentWaypoint() { return currentWaypoint; }


   std::string GetOnScreenDebugString();
   bool CompareSBlobsByDistance(CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob* a, CCI_ColoredBlobOmnidirectionalCameraSensor::SBlob* b);





protected:
   //-- events
   virtual void OnWaypointReached();

   //-- perception
   void GetBlobSensorData(std::vector<float>& dataArray_);

   //-- navigation
   bool GetIsAtWaypoint(Waypoint* wp_);
   bool NavigateTowardsWaypoint(Waypoint* wp_);
   CVector2 CalculateVectorToLight();
   bool UpdateDiffusionVector();
   bool IsSomethingNearby();
   bool NavigateTowardsVector(const CVector2& vector_ );
   void SetWheelSpeedsFromRelativeVector(const CVector2& relVector_);
   void SetWheelSpeedsFromAbsoluteVector(const CVector2& absVector_);
   void SetWheelSpeedsFromDiffusionVector(const CVector2& relVector_);
   void SetWheelSpeeds(Real left_, Real right_);

   float GetColorValue(CColor color_);


   //----------------------------
   GWRNN* gwrNN;
   int numId;
   std::vector<Waypoint*> waypoints;
   Waypoint* currentWaypoint;

   //-- perception
   CCI_DifferentialSteeringActuator* wheels;
   CCI_FootBotProximitySensor* proximitySensor;
   CCI_FootBotLightSensor* lightSensor;
   CCI_LEDsActuator* leds;
   CCI_ColoredBlobOmnidirectionalCameraSensor* blobSensor;
   CCI_FootBotMotorGroundSensor* groundSensor;


   //-- navigation
   CVector3 positionEstimate;
   CVector3 positionTruth;
   CRadians rotationEstimate;
   CRadians rotationTruth;
   CVector2 diffusionVector;

   //-- hard-coded parameters
   CRadians param_collisionAvoidingAngle;

   //-- parameters from the XML file
   struct NavigationParams {
      enum TURNING_MECHANISM {
         NO_TURN = 0, // go straight
         SOFT_TURN,   // both wheels are turning forwards, but at different speeds
         HARD_TURN    // wheels are turning with opposite speeds
      } turningMechanism;
      CRadians hardTurnOnAngleThreshold;
      CRadians noTurnAngleThreshold;
      Real maxSpeed;
      float wheelNoizeStdDev;             //Gaussian noize added to each wheel
      /*
       * Maximum tolerance for the proximity reading between
       *   the robot and the closest obstacle.
       * The proximity reading is 0 when nothing is detected
       * and grows exponentially to 1 when the obstacle is
       * touching the robot.
       */
      Real delta;
      CRange<CRadians> goStraightAngleRange; //-- Angle tolerance range to go straight.
      bool rotateTowardsTargetBehindByLeft = false;
      int angleToTargetChangeCounter = 0;

      void Init(TConfigurationNode& t_tree);
   };
   NavigationParams params_navigation;

   struct SensorsParams {
      float colorBlobMaxRange;            // maximum range that the robot can detect blobs from. The physical restriction is about 90
      float colorBlobMaxAngle;            // maximum angle (in degrees) that the robot can detect blobs from (e.g. use 45 for front 45;-45 range)
      float blobCameraNoizeStdDev;        // Gaussian noise applied to the blob sensors values, including color
      void Init(TConfigurationNode& t_tree);
   };
   SensorsParams params_sensors;


private:



};


#endif
