/* ================================================================
 * ================================================================
 *
 * Implementation of the Grow-When-Required Neural Network.
 *
 * References:
 * [1] Marsland, S., Shapiro, J., & Nehmzow, U. (2002). A self-organising network that grows when required. Neural Networks, 15(8–9), 1041–1058.
 * [2] Marsland, S., Nehmzow, U., & Shapiro, J. (2005). On-line novelty detection for autonomous mobile robots. Robotics and Autonomous Systems, 51(2–3), 191–206.
 * [3] Neto, H. V., & Nehmzow, U. (2007). Real-time automated visual inspection using mobile robots. Journal of Intelligent and Robotic Systems, 49(3), 293–307.
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */

#ifndef CONTROLLERS_GWRNN_H_
#define CONTROLLERS_GWRNN_H_


#include <vector>
#include <argos3/core/utility/logging/argos_log.h>
#include <argos3/core/utility/configuration/argos_configuration.h>

#include "../helpers/helpers.h"

using namespace argos;

class GWRNN {
public:

   struct Connection;
	struct Neuron {

	   int id;                       //for debugging only
	   float habituation;

	   std::vector<float> inWeights; // weights to the input vector // TODO: for now, assumes length of 9 (3 blobs)
	   std::vector<Connection*> connections;


	   Neuron(int id_, float initialHabitation_, int numOfInWeights_) {
	      id = id_;
	      habituation = initialHabitation_;
	      for (int i=0; i<numOfInWeights_; i++) {
	         AddWeight();
	      }

	   }

	   void AddWeight() {
	      //LOGERR << "[] Adding weight to " << Print() << std::endl;
	      inWeights.push_back(Helpers::GetRandomFloat(0,1));
	   }

	   std::string Print() {
	      return Print(false);
	   }
	   std::string Print(bool printWeights_) {
	      return "N" + std::to_string(id) + "[" + std::to_string(inWeights.size()) + "]";
	   }
	};

	struct Connection {
	   int age = 0;
	   Neuron* neuron1;
	   Neuron* neuron2;

	   Connection(Neuron* neuron1_,Neuron* neuron2_) {
	      neuron1 = neuron1_;
	      neuron2 = neuron2_;
	   }

	   bool DoesConnectNeurons(Neuron*neuron1_, Neuron* neuron2_);
	   Neuron* GetNeuronConnectedTo(Neuron* neuron_);
	};

   GWRNN();
   virtual ~GWRNN();
   void Init(argos::TConfigurationNode& t_tree);

   void ProcessInput(std::vector<float> input_);

   //float GetNoveltyValue() { if (winningNode != NULL) { return winningNode->habituation; } else { return 0; } }
   float GetNoveltyValue() { return currentNoveltyValue; }
   int GetWinningNeuronNumber() { return currentWinningNeuron; };
   float GetWinningNeuronError() { return currentWinningNeuronError; }
   float GetWinningNeuronNeighboursError() { return currentWinningNeuronNeighboursError; }
   float GetNeuronActivation(Neuron* neuron_, std::vector<float> input_);
   int GetMaxInputSize() { return params.initialNumOfInWeights; }
   bool GetIsNumOfInWeightsAdaptive() { return params.useAdaptiveInWeightNumber; }

   void SetRobotId(int id_) { robotId = id_; }


protected:

   Neuron* AddNeuron();
   Connection* AddConnection(Neuron* neuron1_, Neuron* neuron2_);
   Connection* GetConnectionBetweenNeurons(Neuron* neuron1_, Neuron* neuron2_);
   void DeleteConnection(Connection* connection_);
   void DeleteNeuron(Neuron* neuron_);

   int robotId;
   int neuronCounter;
   std::vector<Neuron*> neurons;
   std::vector<Connection*> connections;
   Neuron* winningNeuron;

   //-- network type parameters
   struct Params {
      bool produceLogOutput;                          // whether any output will be printed into ARGoS's log

      int initialNumOfInWeights;                      // how many weight from input to itself each neuron has initially
      bool useAdaptiveInWeightNumber;                 // whether the weights away from inputs can grow in size
      float maxConnectionAge;                         // set 0 for infinite age; 50 from Marsland 2002
      bool useDishabituation;                         // send stimulus of 0 to all neurons that are not winning neuron or not connected to it

      //-- behaviour parameters
      float initialNodeHabituation;                   // [2]: 1.0
      float activationThreshold;                      // [2]: 0.7
      float habituationThreshold;                     // [2]: 0.15, [3]:0.3
      float learningRate_winningNode;                 // [2]: 0.3
      //float learningRate_neighbourNodes;            // [2]: 0.15
      float habituationUpdate_S;                      // [2]: 1
      float habituationUpdate_S_nonStimulus;          // [2]: 0
      float habituationUpdate_alpha_winningNeuron;    // [2]: 1.05
      float habituationUpdate_alpha_neighbourNeurons; // [2]: 1.05
      float habituationUpdate_tau_winningNeuron;      // [2]: 3.33
      //float habituationUpdate_tau_neighbourNodes;   // [2]: 14.33
      float updates_neighbourNeuronProportionality;   // [3]: 0.1 - for calculating the learning rate and habituation update of neighbour neurons based on a ratio of activation and winner activation


      void Init(argos::TConfigurationNode& t_tree);
   };
   Params params;

   //-- reporting
   float currentNoveltyValue;
   int currentWinningNeuron;
   float currentWinningNeuronError;
   float currentWinningNeuronNeighboursError;







};

#endif /* CONTROLLERS_GWRNN_H_ */
