/* ================================================================
 * ================================================================
 *
 * Base world object that can be defined in the ARGoS XML.
 * Makes use of prototype entities.
 * Each world object XML has a type attribute, which is used to
 * instantiate a specific subclass of BaseWorldObject.
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */
#include "baseWorldObject.h"


BaseWorldObject::BaseWorldObject(TYPE type_, CPrototypeEntity* prototypeEntity_, const CVector3* arenaSize_) {
   prototypeEntity = prototypeEntity_;
   arenaSize = arenaSize_;
   type = type_;
}

BaseWorldObject::~BaseWorldObject() {

}

/**
 * Convert a type string found in the ARGoS XML file into a TYPE enum member
 */
BaseWorldObject::TYPE BaseWorldObject::GetTypeByString(std::string typeStr_) {
   if (typeStr_ == "GREEN_BOX") {
      return TYPE::GREEN_BOX;
   } else if (typeStr_ == "RED_BOX") {
      return TYPE::RED_BOX;
   } else if (typeStr_ == "CLOSED_DOOR") {
      return TYPE::CLOSED_DOOR;
   } else if (typeStr_ == "OPEN_DOOR") {
      return TYPE::OPEN_DOOR;
   }
   return TYPE::UNKNOWN;
}

/**
 * Convert a TYPE enum member into a type string found in the ARGoS XML file
 */
std::string BaseWorldObject::GetTypeStr() {
   switch (type) {
   case GREEN_BOX:
      return "GREEN_BOX";
   case RED_BOX:
      return "RED_BOX";
   case CLOSED_DOOR:
      return "CLOSED_DOOR";
   case OPEN_DOOR:
      return "OPEN_DOOR";
   default:
      return "UNKNOWN";
   }
}


/* ==================================================================================== */
/* ========================= MOVEMENT */
/* ==================================================================================== */

/**
 * Change the position of the object by deltaX_, deltaY_
 */
void BaseWorldObject::ChangePosition(float deltaX_, float deltaY_) {
   if (prototypeEntity != NULL) {
      //-- generate a new position
      CVector3 position = prototypeEntity->GetEmbodiedEntity().GetOriginAnchor().Position;
      position.SetX(position.GetX() + deltaX_);
      position.SetY(position.GetY() + deltaY_);

      //-- make sure the position is not outside of arena boundaries, which could cause an ARGoS crash
      if (position.GetX() + WORLD_PADDING > arenaSize->GetX()/2) { position.SetX(arenaSize->GetX()/2 - WORLD_PADDING); }
      else if (position.GetX() - WORLD_PADDING < -arenaSize->GetX()/2) { position.SetX(-arenaSize->GetX()/2 + WORLD_PADDING); }

      if (position.GetY() + WORLD_PADDING > arenaSize->GetY()/2) { position.SetY(arenaSize->GetY()/2 - WORLD_PADDING); }
      else if (position.GetY() - WORLD_PADDING < -arenaSize->GetY()/2) { position.SetY(-arenaSize->GetY()/2 + WORLD_PADDING); }

      //-- set the new position
      prototypeEntity->GetEmbodiedEntity().MoveTo(position, CQuaternion());
   }
}




