/* ================================================================
 * ================================================================
 *
 * Various helper functions for string and vector manipulation
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */

#include "helpers.h"
#include "math.h"

#include <iomanip>
#include <sstream>

float Helpers::GetRandomFloat(float min_, float max_) {
   return (float)Helpers::GetInstance().randNumGenerator->Uniform(CRange<Real>(min_, max_));
}

float Helpers::GetRandomGaussian(float stdDev_) {
   return (float)Helpers::GetInstance().randNumGenerator->Gaussian(stdDev_);
}


/* ==================================================================================== */
/* ========================= OUTPUT AND STRINGS */
/* ==================================================================================== */

/**
 * Convert a float number_ to a string with a certain precision_
 */
std::string Helpers::FloatToString(float number_, int precision_) {
   std::stringstream stream;
   stream << std::fixed << std::setprecision(precision_) << number_;
   return (stream.str());
}

/**
 * Returns the inString_, minus everything that's after a number (e.g. node1a -> node)
 */
std::string Helpers::GetNumberlessString(const std::string& inString_) {
   //-- find the id without its numerical affix. Assumes that ids do not have numbers.
   std::size_t found = inString_.find_first_of("0123456789");
   return inString_.substr(0, found);
}


/* ==================================================================================== */
/* ========================= VECTORS */
/* ==================================================================================== */

/**
 * Calculate distance between 2 vectors based on N elements
 * where N=min(vec1size, vec2size)
 */
float Helpers::GetVectorDistance(const std::vector<float>& vec1_, const std::vector<float>& vec2_) {
   int vecSize = Min(vec1_.size(), vec2_.size());
   float totalSum = 0;
   for (int i=0; i<vecSize; i++) {
      totalSum += pow(vec1_[i] - vec2_[i],2);
   }
   return sqrt (totalSum);
}




/* ===================================== LOWER LEVEL VECTOR STUFF ========================*/

/*
 * Get vector to an object translated into local coordinates
 * @param vector_ CRVector3d vector to point towards
 * @param anchorGlobalRotation_: Assuming angle between X and Y, where 0 is pointing along positive Y axis, 90 is pointing along positive X axis
 * @return CRVector3d vector where X component represents angle towards the object and y component distance.
 *    Positive X indicates the target being on the right of the anchor.
 *    Positive Y indicates the target being in front of the anchor.
 */
CVector2 Helpers::ConvertVectorToLocalCoordinateSystem(const CVector2& vector_, double anchorGlobalRotation_) {
   CVector2 tempVec = CVector2();
   tempVec.SetX(vector_.GetX()*cos(-anchorGlobalRotation_) + vector_.GetY()*sin(-anchorGlobalRotation_));
   tempVec.SetY(-vector_.GetX()*sin(-anchorGlobalRotation_) + vector_.GetY()*cos(-anchorGlobalRotation_));
   return tempVec;
}

/**
 * Returns relative vector to something in polar coordinates.
 *
 * @param anchorGlobalPosition_: Global position of the object the relative vector should point from (anchor)
 * @param anchorGlobalRotation_: Assuming angle between X and Y, where 0 is pointing along positive Y axis, 90 is pointing along positive X axis
 * @param targetGlobalPosition_: Global position of the object the relative vector should point to
 *
 * @return CRVector3d vector where X and Y components represent relative vector.
 *    Positive X indicates the target being on the right of the anchor.
 *    Positive Y indicates the target being in front of the anchor.
 */
CVector2 Helpers::GetRelativeVectorFromAnchor(const CVector2& anchorGlobalPosition_, double anchorGlobalRotation_, const CVector2& targetGlobalPosition_) {
   //LOGERR << "--- get rel vector from anchor at " << anchorGlobalPosition_ << " rot " << anchorGlobalRotation_ << "  to " << targetGlobalPosition_ << std::endl;
   CVector2 tempVector = targetGlobalPosition_ - anchorGlobalPosition_;
   CVector2 u = Helpers::ConvertVectorToLocalCoordinateSystem(tempVector,anchorGlobalRotation_);
   if (std::isnan(u.GetX())) {
      u.SetX(0);
   }
   return u;
}

