/* ================================================================
 * ================================================================
 *
 * Base class for representing the environment.
 * Includes functions such as ARGoS XML file parsing, object instantiation,
 * scene initialisation, logging.
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */
#include "baseEnvironment.h"

#include <argos3/core/simulator/simulator.h>
#include <argos3/core/utility/configuration/argos_configuration.h>

#include <argos3/plugins/simulator/visualizations/qt-opengl/qtopengl_render.h>
#include <argos3/plugins/simulator/visualizations/qt-opengl/qtopengl_main_window.h>
#include <argos3/plugins/simulator/visualizations/qt-opengl/qtopengl_widget.h>
#include <argos3/plugins/simulator/visualizations/qt-opengl/qtopengl_user_functions.h>

//#include <QtCore/QVariant>

#include <argos3/core/utility/configuration/tinyxml/ticpp.h>

#include <argos3/plugins/robots/prototype/simulator/prototype_entity.h>
#include <argos3/plugins/robots/prototype/simulator/prototype_link_equipped_entity.h>
#include <argos3/plugins/robots/prototype/simulator/prototype_link_entity.h>
//#include <argos3/plugins/simulator/media/led_medium.h>

#include "../helpers/helpers.h"
#include "../entities/box.h"
#include "../entities/door.h"
#include "../helpers/logger.h"
#include "../controllers/baseRobot.h"
#include "../controllers/gwrNN.h"
#include "../helpers/constants.h"



/*================================== INITIALISATION =================== */

BaseEnvironment::BaseEnvironment() {
   didHandleFirstPreStep = false;
   numOfRobots = 0;
   param_outputFileNamePrefix = "";
}

/**
 * Initialise the environment
 */
void BaseEnvironment::Init(TConfigurationNode& xmlNode_) {

   //-- read parameters
   try {
      //-- parameters
      SetupParameters(xmlNode_);

      //-- create a new RNG
      //randNumGenerator = CRandom::CreateRNG("argos");
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error parsing main loop function parameters", ex);
   }

   //-- read waypoints and give correct ones to the robot
   try {
      int counter = 0;
      float x, y, robotId, robotOrientation;
      Waypoint::TYPE type;
      std::string typeStr, wpId;

      TConfigurationNode& waypointsNode = GetNode(xmlNode_, "waypoints");
      for(ticpp::Element* e = waypointsNode.FirstChildElement("waypoint", false); e != NULL; e = e->NextSiblingElement("waypoint", false)) {
         GetNodeAttribute(*e, "x", x);
         GetNodeAttribute(*e, "y", y);
         GetNodeAttribute(*e, "robotId", robotId);
         //-- try to get type, default to normal
         type = Waypoint::TYPE::NORMAL;
         try {
            GetNodeAttribute(*e, "type", typeStr);
            if (typeStr == "instant") {
               type = Waypoint::TYPE::INSTANT;
            }
         } catch (...) { }

         //-- try to get id, default to the waypoint order
         try {
            GetNodeAttribute(*e, "id", wpId);
         } catch (...) {
            wpId = std::to_string(counter);
         }

         //-- try to get robot orientation, default to invalid (will just position robot to whatever rotation it currently is)
         try {
            GetNodeAttribute(*e, "robotOrientation", robotOrientation);
         } catch (...) {
            robotOrientation = WAYPOINT_INVALID_ORIENTATION;
         }
         waypoints.push_back(Waypoint(wpId,x,y,robotId,type,robotOrientation));
         counter++;
      }
   } catch(CARGoSException& ex) {

   }
   LOG << "Found total of " << waypoints.size() << " waypoints " << std::endl;

   //-- read world object types and store them in a map where object types can be looked up based on numberless object ids
   //   (an alternative where the configuration node is later on obtained from the entity itself doesn't work on MacOS
   std::string type, id;
   try {
      //-- hard-coded prototypes
      ticpp::Node* arenaNode = xmlNode_.Parent()->FirstChild("arena");
      for(ticpp::Element* e = arenaNode->FirstChildElement("prototype", false); e != NULL; e = e->NextSiblingElement("prototype", false)) {
         GetWorldObjectIdAndType(e,id,type);
         //LOG << " prototype " << type << "   " << id << " " <<  std::endl;
         idsToObjectTypes.insert(std::pair<std::string, std::string>(id,type));
      }

      //-- prototypes in the distribute tag
      for(ticpp::Element* e = arenaNode->FirstChildElement("distribute", false); e != NULL; e = e->NextSiblingElement("distribute", false)) {
         ticpp::Node* entityNode = e->FirstChild("entity", false);
         if (entityNode != NULL) {
            ticpp::Element* protypeNode = entityNode->FirstChildElement("prototype", false);
            if (protypeNode != NULL) {
               GetWorldObjectIdAndType(protypeNode,id,type);
               //LOG << " prototype " << type << "   " << id << " " <<  std::endl;
               idsToObjectTypes.insert(std::pair<std::string, std::string>(id,type));
            }
         }
      }
   } catch(CARGoSException& ex) {

   }

   OnNewRun();

}

/**
 * Parse the XML parameters
 */
void BaseEnvironment::SetupParameters(TConfigurationNode& xmlNode_) {
   TConfigurationNode& settingsNode = GetNode(xmlNode_, "settings");
   GetNodeAttribute(settingsNode, "outputFileNamePrefix", param_outputFileNamePrefix);
}

/**
 * Reset the environment
 */
void BaseEnvironment::Reset() {
   outputFileRobotsLog.close();
   Logger::Instance()->CloseOutputFiles();
	OnNewRun();
}

/**
 * Triggered when a new run starts (i.e., simulation starts for the first time or is reset)
 */
void BaseEnvironment::OnNewRun() {

	LOG << "=============== NEW RUN ===============" << std::endl;

	didHandleFirstPreStep = false;

	//-- setup targets
	int i,w;
	double x,y,z;
	//-- attach leds to distinguish between targets
	std::string entityId;
	w = 0;

	//-- iterate through world objects (argos prototypes), create their world object representations in the world objects array
	try {
      CSpace::TMapPerType& targets = GetSpace().GetEntitiesByType("prototype");
      for(CSpace::TMapPerType::iterator it = targets.begin(); it != targets.end(); ++it) {
         try {
            CPrototypeEntity* prototypeEntity = any_cast<CPrototypeEntity*>(it->second);
            std::string type = idsToObjectTypes.find(Helpers::GetNumberlessString(prototypeEntity->GetId()))->second;;
            //LOG << prototypeEntity->GetId() << "  type " << type << std::endl;
            CreateWorldObject(BaseWorldObject::GetTypeByString(type), prototypeEntity);
         } catch (...) {
            //-- type not found - ignore
         }
      }
	} catch (...) { }



	//-- reset the output files
	std::ostringstream fileNameRobotsLog;
	fileNameRobotsLog << param_outputFileNamePrefix << "_robotsLog.txt";
	outputFileRobotsLog.open(fileNameRobotsLog.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileRobotsLog << "time\trobotId\tposX\tposY\tstate\tnoveltyVal\twinningNeuron\tWN error\tWN neighbours error" << std::endl;

	Logger::Instance()->OpenOutputFiles(param_outputFileNamePrefix);

	//-- setup robots
	Logger::Out("Setting up robots...");
	lastRobotWaypointPosition.clear();
	CSpace::TMapPerType& robots = GetSpace().GetEntitiesByType("foot-bot");
	for (i=0; i<robots.size(); i++) {
		std::ostringstream oss;
		oss << "robot" << i;
		CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(robots[oss.str()]);
		BaseRobot& robot = dynamic_cast<BaseRobot&>(footBotEntity.GetControllableEntity().GetController());
		robot.SetId(i);
		for (w=0; w<waypoints.size(); w++) {
		   if (waypoints[w].robotId == i) {
		      robot.AddWaypoint(waypoints[w]);
		   }
		}

		lastRobotWaypointPosition.push_back(CVector3(-9999,-9999,-9999));

	}
	numOfRobots = robots.size();
	Logger::Out("[Setup done]");
}

/**
 * Destroy the environment
 */
void BaseEnvironment::Destroy() {
   outputFileRobotsLog.close();

   Logger::DeleteInstance();
}

/* ============================================== */

/**
 * Draw on the floor
 */
CColor BaseEnvironment::GetFloorColor(const CVector2& position_) {
	//-- draw waypoints
	for(int i = 0; i < waypoints.size(); ++i) {
	   if ((position_ - CVector2(waypoints[i].position.GetX(), waypoints[i].position.GetY())).Length() <=  WAYPOINT_RADIUS) {
	      return CColor::GREEN;
	   } else if (waypoints[i].type == Waypoint::TYPE::INSTANT) {
	      //-- draw coloured outline around instant
         if ((position_ - CVector2(waypoints[i].position.GetX(), waypoints[i].position.GetY())).Length() <=  WAYPOINT_RADIUS + 0.05) {
            return CColor::ORANGE;
         }
      }


	}
	return CColor::WHITE;
}

/* ==================================================================================== */
/* ========================= Update loop */
/* ==================================================================================== */

/**
 * Executed at each update loop, before robot's update loop step is executed.
 */
void BaseEnvironment::PreStep() {
   Logger::Instance()->SetCurrentSimulationTime(GetSpace().GetSimulationClock());

   if (!didHandleFirstPreStep) {
      OnFirstPreStep();
      didHandleFirstPreStep = true;
   }


	//-- go through all robots
	CSpace::TMapPerType& m_cFootbots = GetSpace().GetEntitiesByType("foot-bot");
	for(CSpace::TMapPerType::iterator it = m_cFootbots.begin(); it != m_cFootbots.end(); ++it) {

      CFootBotEntity& robotEntity = *any_cast<CFootBotEntity*>(it->second);
      BaseRobot& robotController = dynamic_cast<BaseRobot&>(robotEntity.GetControllableEntity().GetController());
      CVector3 robotPosition = robotEntity.GetEmbodiedEntity().GetOriginAnchor().Position;

      Waypoint* currentWp = robotController.GetCurrentWaypoint();
      if (currentWp != NULL) {
         //-- teleport robot to instant waypoints, either using waypoint's robotOrientation or robot's orientation if waypoint doesn't specify
         if (currentWp->type == Waypoint::TYPE::INSTANT) {
            try {
               if (currentWp->robotOrientation == WAYPOINT_INVALID_ORIENTATION) {
                  robotEntity.GetEmbodiedEntity().MoveTo(currentWp->position, robotEntity.GetEmbodiedEntity().GetOriginAnchor().Orientation);
               } else {
                  CQuaternion quaternion;
                  CRadians angle;
                  angle.FromValueInDegrees(currentWp->robotOrientation);
                  quaternion.FromEulerAngles(angle, CRadians(0), CRadians(0));
                  robotEntity.GetEmbodiedEntity().MoveTo(currentWp->position, quaternion);
               }
            } catch (...) { }
         }

         //-- fire on waypoint event
         if ((robotPosition - currentWp->position).Length() <= WAYPOINT_RADIUS) {
            OnRobotAtCurrentWaypoint(robotEntity, robotController, currentWp);
         }
      }

      //-- give robot GPS information
      robotController.SetPositionTruth(robotPosition);
      robotController.SetPositionEstimate(robotPosition);

      //-- give robot compass information
      CRadians xAngle, yAngle, zAngle;
      robotEntity.GetEmbodiedEntity().GetOriginAnchor().Orientation.ToEulerAngles(zAngle, yAngle, xAngle);
      robotController.SetRotationTruth(-zAngle + CRadians(M_PI/2));
      robotController.SetRotationEstimate(-zAngle + CRadians(M_PI/2));

      //-- output robot info to log:
      Real averageTimeToReachWorksites = 0;
      outputFileRobotsLog << GetSpace().GetSimulationClock() - 1 << "\t" //do -1 because this is a pre-step
             << robotController.GetNumId() << "\t"
             << robotPosition.GetX() << "\t" << robotPosition.GetY() << "\t"
             << -1 << "\t"
             << robotController.GetNetwork()->GetNoveltyValue() << "\t"
             << robotController.GetNetwork()->GetWinningNeuronNumber() << "\t"
             << robotController.GetNetwork()->GetWinningNeuronError() << "\t"
             << robotController.GetNetwork()->GetWinningNeuronNeighboursError() << "\t"
             << std::endl;

   }
}

/* ==================================================================================== */
/* ========================= Events */
/* ==================================================================================== */


/**
 * Triggered every time a robot is at its current waypoint (a robot might not realise it yet)
 */
void BaseEnvironment::OnRobotAtCurrentWaypoint(CFootBotEntity& robotEntity_, BaseRobot& robotController_, Waypoint* currentWp_) {
   //LOG << " robot at cur waypoint " << currentWp_->position << std::endl;
   if (lastRobotWaypointPosition[robotController_.GetNumId()] != currentWp_->position) {
      OnRobotFirstTimeAtCurrentWaypoint(robotEntity_, robotController_, currentWp_);
   }
   lastRobotWaypointPosition[robotController_.GetNumId()] = CVector3(currentWp_->position);
}

/**
 * Triggered first time a robot is at its current waypoint (a robot might not realise it yet)
 */
void BaseEnvironment::OnRobotFirstTimeAtCurrentWaypoint(CFootBotEntity& robotEntity_, BaseRobot& robotController_, Waypoint* currentWp_) {
   if (currentWp_->type == Waypoint::TYPE::INSTANT) {
      Logger::Instance()->LogEvent(Event::TYPE::ROBOT_TELEPORTED, robotController_.GetNumId(), currentWp_->position.GetX(), currentWp_->position.GetY());
   }
   //if (currentWp_->id == WAYPOINT_END_LOOP_ID) {
   Logger::Instance()->LogEvent(Event::TYPE::ROBOT_REACHED_WAYPOINT, robotController_.GetNumId(), currentWp_->position.GetX(), currentWp_->position.GetY(), currentWp_->id);
   //}
}


/* ==================================================================================== */
/* ========================= Other */
/* ==================================================================================== */

/**
 * Create a world object of a specific class based on type
 */
void BaseEnvironment::CreateWorldObject(BaseWorldObject::TYPE type_, CPrototypeEntity* prototypeEntity_) {
   switch (type_) {
      case BaseWorldObject::TYPE::GREEN_BOX:
      case BaseWorldObject::TYPE::RED_BOX: {
         Box* box = new Box(type_, prototypeEntity_, &GetSpace().GetArenaSize());
         box->SetType(type_);
         worldObjects.push_back(box);
         } break;
      case BaseWorldObject::TYPE::CLOSED_DOOR:
      case BaseWorldObject::TYPE::OPEN_DOOR: {
         Door* door = new Door(type_, prototypeEntity_, &GetSpace().GetArenaSize());
         door->SetType(type_);
         worldObjects.push_back(door);
         }break;
      default:
         Logger::OutErr("Cannot create object of type " + std::to_string(type_));
         break;
   }
}

/**
 * Determine numberless id and type of a world object defined by xml prototype node.
 * Type can either be found on the <links> subnode, which is default.
 * Type can also be overwritten if defined on the node_ itself.
 */
void BaseEnvironment::GetWorldObjectIdAndType(ticpp::Element* node_, std::string& id_, std::string& type_) {
   //-- id
   GetNodeAttribute(*node_,"id",id_);
   id_ = Helpers::GetNumberlessString(id_);

   //-- try type on the node_ itself
   try {
      GetNodeAttribute(*node_, "type", type_);
   } catch (...) {
      //-- has to be on the <links> subnode
      ticpp::Element* linksE = node_->FirstChildElement("links", false);
      if (linksE != NULL) {
         GetNodeAttribute(*linksE,"type",type_);
      }
   }
}

/**
 * Returns a robot that has a certain id
 */
BaseRobot& BaseEnvironment::GetRobotById(const int& id_) {
	CSpace::TMapPerType& robots = GetSpace().GetEntitiesByType("foot-bot");
	std::ostringstream oss;
	oss << "robot" << id_;
	CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(robots[oss.str()]);
	BaseRobot& robot = dynamic_cast<BaseRobot&>(footBotEntity.GetControllableEntity().GetController());
	return robot;

}


/* ==================================================================================== */
/* ========================= Register with ARGoS */
/* ==================================================================================== */

REGISTER_LOOP_FUNCTIONS(BaseEnvironment, "baseEnvironment")
