/* ================================================================
 * ================================================================
 *
 * Represents the Corridor environment modelled based on
 * Marsland, S., Nehmzow, U., & Shapiro, J. (2005). On-line novelty detection for autonomous mobile robots. Robotics and Autonomous Systems, 51(2–3), 191–206.
 *
 * The environment contains walls and closed doors (see entities/door.h).
 * Door and light positions are randomised when the simulation starts.
 * After the robot traversed the corridor N times, a random door is opened.
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */


#include "corridorEnvironment.h"

#include <string>

#include "../controllers/baseRobot.h"
#include "../helpers/helpers.h"
#include "../helpers/logger.h"

#include "../../cppCommon/event.h"

CorridorEnvironment::CorridorEnvironment() {
   robotLearningLoopsCounter = 0;
   param_numOfLearningLoopsBeforeChange = 0;
   doorToOpen = 0;
}

CorridorEnvironment::~CorridorEnvironment() {

}


/* ==================================================================================== */
/* ========================= Initialisation */
/* ==================================================================================== */

/**
 * Process XML for parameters
 */
void CorridorEnvironment::SetupParameters(TConfigurationNode& xmlNode_) {
   BaseEnvironment::SetupParameters(xmlNode_);

   TConfigurationNode& settingsNode = GetNode(xmlNode_, "settings");
   GetNodeAttribute(settingsNode, "numOfLearningLoopsBeforeChange", param_numOfLearningLoopsBeforeChange);
}

/**
 * Triggered when the simulation starts, before a robot's first update loop is run
 */
void CorridorEnvironment::OnFirstPreStep() {
   robotLearningLoopsCounter = 0;

   BaseEnvironment::OnFirstPreStep();

   //-- for each door, move it randomly by +/- N on y axis
   try {
      CSpace::TMapPerType& prototypes = GetSpace().GetEntitiesByType("prototype");
      for (int doorId=0; doorId<prototypes.size(); doorId++) {
         std::ostringstream oss;
         oss << "door" << doorId;
         CPrototypeEntity& doorEntity = *any_cast<CPrototypeEntity*>(prototypes[oss.str()]);
         doorEntity.GetEmbodiedEntity().MoveTo(CVector3(doorEntity.GetEmbodiedEntity().GetOriginAnchor().Position.GetX(),
               doorEntity.GetEmbodiedEntity().GetOriginAnchor().Position.GetY() + Helpers::GetRandomFloat(-1.5, 1.5), doorEntity.GetEmbodiedEntity().GetOriginAnchor().Position.GetZ()),
               doorEntity.GetEmbodiedEntity().GetOriginAnchor().Orientation);
      }
   } catch (CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error distributing doors", ex);
   }

   Logger::OutEnvironment("Pass " + std::to_string(robotLearningLoopsCounter) + " Change at " + std::to_string(param_numOfLearningLoopsBeforeChange));

}



/* ==================================================================================== */
/* ========================= Events */
/* ==================================================================================== */

/**
 * Handle environment change when robot reached a certain waypoint
 */
void CorridorEnvironment::OnRobotFirstTimeAtCurrentWaypoint(CFootBotEntity& robotEntity_, BaseRobot& robotController_, Waypoint* currentWp_) {
   BaseEnvironment::OnRobotFirstTimeAtCurrentWaypoint(robotEntity_, robotController_, currentWp_);
   if (robotController_.GetNumId() == 0 && currentWp_->id == "endLoop") {
      robotLearningLoopsCounter ++;

      //-- after robot has gone through the corridor N times, tell one of the "doors" to "open"
      if (robotLearningLoopsCounter == param_numOfLearningLoopsBeforeChange) {
         doorToOpen = round(Helpers::GetRandomFloat(0, 4));
         Logger::Out("========= OPENING DOOR " + std::to_string(doorToOpen));
         worldObjects[doorToOpen]->SetType(BaseWorldObject::TYPE::OPEN_DOOR);

         //-- log the change
         Logger::Instance()->LogEvent(Event::TYPE::ENV_CHANGED,doorToOpen,worldObjects[doorToOpen]->GetPosition().GetX(),worldObjects[doorToOpen]->GetPosition().GetY());
      }

      //-- output into the 2D overlay
      if (robotLearningLoopsCounter < param_numOfLearningLoopsBeforeChange) {
         Logger::OutEnvironment("Pass " + std::to_string(robotLearningLoopsCounter) + " Change at " + std::to_string(param_numOfLearningLoopsBeforeChange));
      }  else {
         Logger::OutEnvironment("Pass " + std::to_string(robotLearningLoopsCounter) + " Change at " + std::to_string(param_numOfLearningLoopsBeforeChange) + "\nOpened door " + std::to_string(doorToOpen));
      }
   }
};



/* ==================================================================================== */
/* ========================= Register with ARGoS */
/* ==================================================================================== */

REGISTER_LOOP_FUNCTIONS(CorridorEnvironment, "corridorEnvironment")
