/* ================================================================
 * ================================================================
 *
 * Represents the Room environment modelled based on
 * Neto, H. V., & Nehmzow, U. (2007). Real-time automated visual inspection using mobile robots. Journal of Intelligent and Robotic Systems, 49(3), 293–307.
 *
 * The environment contains walls around and in the middle.
 * Wall light positions are randomised when the simulation starts.
 * After the robot traversed the room N times, a box is placed in a random corner of the room.
 *
 * Author: L. Pitonakova (http://lenkaspace.net)
 * License: GNU General Public License. Please credit me when using my work.
 *
 * ================================================================
 * ================================================================
 */

#include "roomEnvironment.h"

#include <string>

#include "../controllers/baseRobot.h"
#include "../helpers/helpers.h"
#include "../helpers/logger.h"

#include "../../cppCommon/event.h"

#include <argos3/plugins/robots/prototype/simulator/prototype_entity.h>

RoomEnvironment::RoomEnvironment() {
   robotLearningLoopsCounter = 0;
   param_numOfLearningLoopsBeforeChange = 0;
   moveDone = false;
}

RoomEnvironment::~RoomEnvironment() {

}



/* ==================================================================================== */
/* ========================= Initialisation */
/* ==================================================================================== */

/**
 * Process XML for parameters
 */
void RoomEnvironment::SetupParameters(TConfigurationNode& xmlNode_) {
   BaseEnvironment::SetupParameters(xmlNode_);

   TConfigurationNode& settingsNode = GetNode(xmlNode_, "settings");
   GetNodeAttribute(settingsNode, "numOfLearningLoopsBeforeChange", param_numOfLearningLoopsBeforeChange);
}

/**
 * Triggered when the simulation starts, before a robot's first update loop is run
 */
void RoomEnvironment::OnFirstPreStep() {
   robotLearningLoopsCounter = 0;
   BaseEnvironment::OnFirstPreStep();
   Logger::OutEnvironment("Pass " + std::to_string(robotLearningLoopsCounter) + " Change at " + std::to_string(param_numOfLearningLoopsBeforeChange));
}




/* ==================================================================================== */
/* ========================= Events */
/* ==================================================================================== */

/**
 * Handle environment change when robot reached a certain waypoint
 */
void RoomEnvironment::OnRobotFirstTimeAtCurrentWaypoint(CFootBotEntity& robotEntity_, BaseRobot& robotController_, Waypoint* currentWp_) {
   BaseEnvironment::OnRobotFirstTimeAtCurrentWaypoint(robotEntity_, robotController_, currentWp_);
   if (robotController_.GetNumId() == 0 && currentWp_->id == "endLoop") {
      robotLearningLoopsCounter ++;
      Logger::OutEnvironment("Pass " + std::to_string(robotLearningLoopsCounter) + " Change at " + std::to_string(param_numOfLearningLoopsBeforeChange));
      //-- after robot has gone through the corridor N times, tell one of the "doors" to "open"
      if (robotLearningLoopsCounter >= param_numOfLearningLoopsBeforeChange && !moveDone) {
         Logger::Out("========= ADDING NEW OBJECT");
         CSpace::TMapPerType& targets = GetSpace().GetEntitiesByType("prototype");
         for(CSpace::TMapPerType::iterator it = targets.begin(); it != targets.end(); ++it) {
            try {
               CPrototypeEntity* prototypeEntity = any_cast<CPrototypeEntity*>(it->second);
               if (prototypeEntity->GetId() == "box") {
                  CVector3 newPos = CVector3(1.6,1.6,0);
                  if (Helpers::GetRandomFloat(0, 1) > 0.5) { newPos.SetX(-1.6); }
                  //if (Helpers::GetRandomFloat(0, 1) > 0.5) { newPos.SetY(-1.6); }
                  prototypeEntity->GetEmbodiedEntity().MoveTo(newPos, prototypeEntity->GetEmbodiedEntity().GetOriginAnchor().Orientation);
                  moveDone = true;

                  //-- log the change
                  Logger::Instance()->LogEvent(Event::TYPE::ENV_CHANGED,-1,prototypeEntity->GetEmbodiedEntity().GetOriginAnchor().Position.GetX(),prototypeEntity->GetEmbodiedEntity().GetOriginAnchor().Position.GetY());
               }
            } catch (...) {
               moveDone = false;
            }
         }
      }
   }
};



/* ==================================================================================== */
/* ========================= Register with ARGoS */
/* ==================================================================================== */

REGISTER_LOOP_FUNCTIONS(RoomEnvironment, "roomEnvironment")

